<?php
/**
 * Sets up custom filters and actions for the theme.  This does things like sets up sidebars, menus, scripts, 
 * and lots of other awesome stuff that WordPress themes do.
 */

/* Register custom image sizes. */
add_action( 'init', 'swt_register_image_sizes', 5 );

/* Register custom menus. */
add_action( 'init', 'swt_register_menus', 5 );

/* Register sidebars. */
add_action( 'widgets_init', 'swt_register_sidebars', 5 );

/* Add custom scripts. */
add_action( 'wp_enqueue_scripts', 'swt_enqueue_scripts' );

/* Register custom styles. */
add_action( 'wp_enqueue_scripts', 'swt_register_styles', 0 );

/* Filters the excerpt length. */
add_filter( 'excerpt_length', 'swt_excerpt_length' );

/* Remove read more in excerpt */
add_filter( 'excerpt_more', 'swt_read_more' );

/* Main theme layout */
add_filter( 'get_theme_layout', 'main_theme_layout' );

/* Adds custom attributes to the subsidiary sidebar. */
add_filter( 'hybrid_attr_sidebar', 'swt_sidebar_subsidiary_class', 10, 2 );

/* Additional css classes for widgets */
add_filter( 'dynamic_sidebar_params', 'swt_widget_classes' );

/* Disable sidebar if 1 column layout */
add_filter( 'sidebars_widgets', 'swt_disable_sidebars' );
add_action( 'template_redirect', 'swt_one_column' );	

/* Remove allowed tags from comment form */
add_filter( 'comment_form_defaults', 'remove_comment_form_allowed_tags' );

/* Remove gallery inline styles */
add_filter( 'use_default_gallery_style', '__return_false' );

/* Extra classes to body and posts. */
add_filter( 'body_class', 'extra_classes' );
add_filter( 'post_class', 'extra_post_classes' );

/* Add header scripts, custom css and favicon to <head> */
add_action( 'wp_head', 'swt_head_hooks' );

/* Add footer scripts to </body> */
add_action( 'wp_footer', 'swt_footer_scripts' ); 
 
/**
 * Registers custom image sizes for the theme.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_image_sizes() {
	
	/* Sets the 'post-thumbnail' size. */
	set_post_thumbnail_size( 150, 150, true );

	/* Widget image size */
	add_image_size( 'swt-widget-image', 90, 90, true );   

	/* Cat image size */
	add_image_size( 'swt-cat-image', 165, 215, true );   

	/* SWT medium size */
	add_image_size( 'swt-medium', 220, 165, true );        	

	/* SWT large size */
	add_image_size( 'swt-large', 750, 533, true );        	
	
	/* Featured widget image */
	add_image_size( 'swt-featured-image', 360, 256, true );        
	
}

/**
 * Registers nav menu locations.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_menus() {
	register_nav_menu( 'primary',    _x( 'Primary',    'nav menu location', 'swt' ) );
}

/**
 * Registers sidebars.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_sidebars() {
 
	hybrid_register_sidebar(
		array(
			'id'          => 'primary',
			'name'        => _x( 'Primary', 'sidebar', 'swt' ),
			'description' => __( 'The main sidebar. It is displayed on either the left or right side of the page based on the chosen layout.', 'swt' )
		)
	);
	
	hybrid_register_sidebar(
		array(
			'id'          => 'featured',
			'name'        => _x( 'Featured', 'sidebar', 'swt' ),
			'description' => __( 'A sidebar located before the content area of site. Optimized for one, two, three or four widgets (and multiples thereof).', 'swt' )
		)
	);	

	hybrid_register_sidebar(
		array(
			'id'          => 'subsidiary',
			'name'        => _x( 'Subsidiary', 'sidebar', 'swt' ),
			'description' => __( 'A sidebar located in the footer of the site. Optimized for one, two, three or four widgets (and multiples thereof).', 'swt' )
		)
	);
}

/**
 * Enqueues scripts.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_enqueue_scripts() {

	wp_register_script( 'swt-responsive-nav', trailingslashit( get_template_directory_uri() ) . 'js/responsive-nav.min.js', array( 'jquery' ), null, true );
	wp_enqueue_script( 'swt-responsive-nav' );

	wp_register_script( 'swt-fitvids', trailingslashit( get_template_directory_uri() ) . 'js/fitvids.js', array( 'jquery' ), null, true );
	wp_enqueue_script( 'swt-fitvids' );

	wp_register_script( 'swt-custom-js', trailingslashit( get_template_directory_uri() ) . 'js/custom.js', array( 'jquery' ), null, true );
	wp_enqueue_script( 'swt-custom-js' );
	

	$ticker = swt_get_option( 'ticker' );	

	wp_register_script( 'swt-slider', trailingslashit( get_template_directory_uri() ) . 'js/slider.js', array( 'jquery' ), null, true );
	
	global $sticky, $sticky_slider;
	
	$sticky = get_option( 'sticky_posts' );
    $sticky_slider = swt_get_option( 'swt_slider' );	
	
	/* Enqueue slider.js if slider activated and blog has sticky posts. */
	if ( $sticky_slider == 1 && ! empty( $sticky ) && ( is_home() || is_front_page() ) ) {

		wp_enqueue_script( 'swt-slider' );

		$slider_mode 	= swt_get_option( 'slider_mode', 'fade' ); 
		$slider_auto 	= swt_get_option( 'slider_auto', 'true' ); 
		$slider_pause 	= swt_get_option( 'slider_speed', 4000 );

		$params = array('mode' 	=> $slider_mode,
						'auto' 	=> $slider_auto,
						'pause' => $slider_pause
		);	

		wp_localize_script( 'swt-custom-js', 'ParamsSlider', $params ); 

	}	

	$fc_slider = swt_get_option( 'fc_slider', false );
	
	if ( $fc_slider ) {		
		$fc_speed = swt_get_option( 'fc_speed', 4 ) * 1000;		
		$params = array( 'fc_speed' => $fc_speed );
		wp_localize_script( 'swt-custom-js', 'ParamsFc', $params );
		
		wp_register_script( 'swt-sly-js', trailingslashit( get_template_directory_uri() ) . 'js/sly.min.js', array( 'jquery' ), null, true );
		wp_enqueue_script( 'swt-sly-js' );		
	}

	$ticker = swt_get_option( 'ticker', false );
	
	if ( $ticker ) {		
		wp_localize_script( 'swt-custom-js', 'ParamsTicker', '' );
 		wp_enqueue_script( 'swt-slider' );
	}			

	if ( !is_singular() ) {

		$masonry = array( 
			'msgfinish' => swt_get_option( 'finishedMsg', 'All Loaded!' ),
			'img'       => trailingslashit( get_template_directory_uri() ) . 'images/ajax-loader.gif' 
		);

		wp_localize_script( 'swt-custom-js', 'swtMasonry', $masonry );
		
		wp_enqueue_script( 'swt-imagesloaded', trailingslashit( get_template_directory_uri() ) . 'js/imagesloaded.pkgd.min.js', array( 'jquery' ), null, true );		
		wp_enqueue_script( 'masonry' );		

		/* Infinite Scroll */
		wp_enqueue_script( 'infinite_scroll',  
							trailingslashit( get_template_directory_uri() ). 'js/infinitescroll.min.js', array( 'jquery' ), null, true );
		wp_enqueue_script( 'manual_trigger', trailingslashit( get_template_directory_uri() ) . 'js/manual-trigger.js', array( 'infinite_scroll' ), null, true );	
			
		add_action( 'wp', 'custom_paged_404_fix' );				
	}
	
}

function custom_paged_404_fix( ) {
	global $wp_query;

	if ( is_404() || !is_paged() || 0 != count( $wp_query->posts ) )
		return;

	$wp_query->set_404();
	status_header( 404 );
	nocache_headers();
}

/**
 * Registers custom stylesheets for the front end.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_styles() {

	/* Font Icon Font */
	wp_register_style( 'font-awesome', '//netdna.bootstrapcdn.com/font-awesome/4.2.0/css/font-awesome.css' );	
	wp_enqueue_style( 'font-awesome' );
	
	/* Theme Fonts */
	wp_register_style( 'font-pt-sans', 'http://fonts.googleapis.com/css?family=PT+Sans:400,700,400italic' );	
	wp_enqueue_style( 'font-pt-sans' );	
	wp_register_style( 'font-alfa-slab', 'http://fonts.googleapis.com/css?family=Alfa+Slab+One' );	
	wp_enqueue_style( 'font-alfa-slab' );		
}
 

/**
 * Adds a custom excerpt length.
 *
 * @since  1.0.0
 * @access public
 * @param  int     $length
 * @return int
 */
function swt_excerpt_length( $length ) {
	return 60;
}

function swt_read_more( $more ) {
	global $post;
	return '';	
}

/*
* Default layout
*/
function main_theme_layout( $layout ) {

	if ( is_singular() )
		$layout = swt_get_option( 'site_layout', 'layout-2c-l' );
	
	return $layout;
}

/**
 * Adds a custom class to the 'subsidiary' sidebar.  This is used to determine the number of columns used to 
 * display the sidebar's widgets.  This optimizes for 1, 2, 3 and 4 columns or multiples of those values.
 *
 * Note that we're using the global $sidebars_widgets variable here. This is because core has marked 
 * wp_get_sidebars_widgets() as a private function. Therefore, this leaves us with $sidebars_widgets for 
 * figuring out the widget count.
 * @link http://codex.wordpress.org/Function_Reference/wp_get_sidebars_widgets
 *
 * @since  1.0.0
 * @access public
 * @param  array  $attr
 * @param  string $context
 * @return array
 */
function swt_sidebar_subsidiary_class( $attr, $context ) {

	if ( 'subsidiary' === $context || 'featured' === $context ) {
		global $sidebars_widgets;

		if ( is_array( $sidebars_widgets ) && !empty( $sidebars_widgets[ $context ] ) ) {

			$count = count( $sidebars_widgets[ $context ] );
			
			if ( ( $count == 4 ) || $count > 4 )
				$attr['class'] .= ' sidebar-col-4';
				
			elseif ( !( $count % 3 ) )
				$attr['class'] .= ' sidebar-col-3';				

			elseif ( !( $count % 2 ) )
				$attr['class'] .= ' sidebar-col-2';

			else
				$attr['class'] .= ' sidebar-col-1';

			$attr['class'] .= ' wrap';
			
		}
	}

	return $attr;
}
 
   
/**
 * Adding .widget-first and .widget-last classes to widgets.
 * Class .widget-last used to reset margin-right to zero in subsidiary sidebar for the last widget.
 *
 * @since 0.1.0
 */
function swt_widget_classes( $params ) {

	global $my_widget_num; // Global a counter array
	$this_id = $params[0]['id']; // Get the id for the current sidebar we're processing
	$arr_registered_widgets = wp_get_sidebars_widgets(); // Get an array of ALL registered widgets

	if ( !$my_widget_num ) { // If the counter array doesn't exist, create it
		$my_widget_num = array();
	}

	if ( !isset( $arr_registered_widgets[$this_id] ) || !is_array( $arr_registered_widgets[$this_id] ) ) { // Check if the current sidebar has no widgets
		return $params; // No widgets in this sidebar... bail early.
	}

	if ( isset( $my_widget_num[$this_id] ) ) { // See if the counter array has an entry for this sidebar
		$my_widget_num[$this_id] ++;
	} else {  // If not, create it starting with 1
		$my_widget_num[$this_id] = 1;
	}

	$class = 'class="widget-' . $my_widget_num[$this_id] . ' ';  // Add a widget number class for additional styling options

	if ( $my_widget_num[$this_id] == 1 ) {  // If this is the first widget
		$class .= 'widget-first ';
	} elseif( $my_widget_num[$this_id] == count( $arr_registered_widgets[$this_id] ) ) { // If this is the last widget
		$class .= 'widget-last ';
	}

	$params[0]['before_widget'] = str_replace( 'class="', $class, $params[0]['before_widget'] ); // Insert our new classes into "before widget"

	return $params;

}

/**
 * Disables sidebars if viewing a one-column page.
 *
 * @since 0.1.0
 */
function swt_disable_sidebars( $sidebars_widgets ) {
	
	global $wp_query;

	if ( current_theme_supports( 'theme-layouts' ) ) {

		if ( 'layout-1c' == theme_layouts_get_layout() ) {	
			$sidebars_widgets['primary'] = false;
		}
                
	}

	return $sidebars_widgets;
}


/**
 * Function for deciding which pages should have a one-column layout.
 *
 * @since 0.1.0
 */ 
function swt_one_column() {

	if ( !is_singular() || !is_active_sidebar( 'primary' ) || is_attachment() && wp_attachment_is_image() )
		add_filter( 'get_theme_layout', 'swt_theme_layout_one_column' );

	elseif ( is_attachment() && 'layout-default' == theme_layouts_get_layout() )
		add_filter( 'get_theme_layout', 'swt_theme_layout_one_column' );
}


/**
 * Filters 'get_theme_layout' by returning 'layout-1c'.
 *
 * @since 0.2.0
 */
function swt_theme_layout_one_column( $layout ) {
	return 'layout-1c';
}

/**
 * Use full image if 1 column layout.
 *
 * @since 0.1.0
 */ 

/* Disables comments allowed tags */
function remove_comment_form_allowed_tags( $defaults ) {
 
	$defaults['comment_notes_before'] = '';
	$defaults['comment_notes_after'] = '';
	return $defaults;
	 
}
 

/*this function allows for the auto-creation of post excerpts*/
function truncate_post( $amount ) {
	$truncate = get_the_content();
	$truncate = apply_filters('the_content', $truncate);
	$truncate = preg_replace('@<script[^>]*?>.*?</script>@si', '', $truncate);
	$truncate = preg_replace('@<style[^>]*?>.*?</style>@si', '', $truncate);
	$truncate = strip_tags($truncate);
	$truncate = substr($truncate, 0, strrpos(substr($truncate, 0, $amount), ' '));
	echo $truncate;
	echo "...";
} 

/* Shortens time words to their abbreviations. */
function swt_time_ago() {

	$replace = array(
		'minute'  => 'min',
		'minutes' => 'mins',
		'hour'    => 'hr',
		'hours'   => 'hrs',
		'day'     => 'd',
		'days'    => 'ds',
		'week'    => 'wk',
		'weeks'   => 'wks',
		'month'   => 'mo',
		'months'  => 'mos',
		'year'    => 'yr',
		'years'   => 'yrs'
	);

	$since = human_time_diff( get_the_time('U'), current_time('timestamp') );
	return strtr( $since, $replace );
}

/* Short titles for news ticker */
function short_title() {
	
	$title = get_the_title();
	$length = strlen( $title );
	
	if ( $length > 37 ) 
		echo substr( $title, 0, 34 ) .'...';			
	else 
		echo $title;					
}

/* Adds extra classes to body. */
function extra_classes( $classes ) {
	
	if ( swt_get_option( 'swt_social_profiles', false ) )
		$classes[] = 'social-icons-on';

	if ( swt_get_option( 'header_banner', false ) )
		$classes[] = 'banner-on';


	if ( swt_get_option( 'top_search', false ) )
		$classes[] = 'top-search-on';	
	
	return $classes;
}

/* Adds extra classes to post. */
function extra_post_classes( $classes ) {

	if ( !is_singular() )
		$classes[] = 'masonry-brick';
	
	return $classes;
}

function swt_human_time_diff( $from, $to = '' ) {
	if ( empty( $to ) ) {
		$to = time();
	}

	$diff = (int) abs( $to - $from );

	if ( $diff < HOUR_IN_SECONDS ) {
		$mins = round( $diff / MINUTE_IN_SECONDS );
		if ( $mins <= 1 )
			$mins = 1;
		/* translators: min=minute */
		$since = sprintf( _n( '%s min', '%s mins', $mins ), $mins );
	} elseif ( $diff < DAY_IN_SECONDS && $diff >= HOUR_IN_SECONDS ) {
		$hours = round( $diff / HOUR_IN_SECONDS );
		if ( $hours <= 1 )
			$hours = 1;
		$since = sprintf( _n( '%s hr', '%s hrs', $hours ), $hours );
	} elseif ( $diff < WEEK_IN_SECONDS && $diff >= DAY_IN_SECONDS ) {
		$days = round( $diff / DAY_IN_SECONDS );
		if ( $days <= 1 )
			$days = 1;
		$since = sprintf( _n( '%s d', '%s ds', $days ), $days );
	} elseif ( $diff < 30 * DAY_IN_SECONDS && $diff >= WEEK_IN_SECONDS ) {
		$weeks = round( $diff / WEEK_IN_SECONDS );
		if ( $weeks <= 1 )
			$weeks = 1;
		$since = sprintf( _n( '%s wk', '%s wks', $weeks ), $weeks );
	} elseif ( $diff < YEAR_IN_SECONDS && $diff >= 30 * DAY_IN_SECONDS ) {
		$months = round( $diff / ( 30 * DAY_IN_SECONDS ) );
		if ( $months <= 1 )
			$months = 1;
		$since = sprintf( _n( '%s mo', '%s mos', $months ), $months );
	} elseif ( $diff >= YEAR_IN_SECONDS ) {
		$years = round( $diff / YEAR_IN_SECONDS );
		if ( $years <= 1 )
			$years = 1;
		$since = sprintf( _n( '%s yr', '%s yrs', $years ), $years );
	}
}
 
/* Exclude sticky posts from home page. */
add_action( 'pre_get_posts', 'swt_exclude_sticky' );

/**
 * Excluding sticky posts from home page if slider enabled. Sticky posts are in a slider.
 * 
 * @since 0.1
 */
function swt_exclude_sticky( $query ) {
	
	/* Exclude if is home and is main query. */
	if ( is_home() && $query->is_main_query() && swt_get_option( 'swt_slider' ) == 1 )
		$query->set( 'post__not_in', get_option( 'sticky_posts' ) );
	
}


/* Functions from theme options - header scripts, custom css and favicon */
function swt_head_hooks() {
	
	$header_scripts = swt_get_option( 'header_scripts' ); 
	$custom_css = swt_get_option( 'custom_css' );
	$favicon = swt_get_option( 'favicon' ); 
	$output = '';
	
	if ( $favicon ) { 
		$output .= "<link rel=\"shortcut icon\" href=\"{$favicon}\" />\n";
	}		
	
	if ( $header_scripts ) 
		$output .= $header_scripts ."\n";
	
	if ( $custom_css ) 
		$output .= "<style type=\"text/css\">\n". $custom_css ."\n</style>\n";
	
	echo $output; 
}

/* Adds footer scripts */
function swt_footer_scripts() {
	
	$footer_scripts = swt_get_option( 'footer_scripts' ); 
	
	if ( $footer_scripts )
		echo $footer_scripts . "\n";

}