<?php
/**
 * Sets up custom filters and actions for the theme.  This does things like sets up sidebars, menus, scripts, 
 * and lots of other awesome stuff that WordPress themes do.
 */

/* Register custom image sizes. */
add_action( 'init', 'swt_register_image_sizes', 5 );

/* Register custom menus. */
add_action( 'init', 'swt_register_menus', 5 );

/* Register sidebars. */
add_action( 'widgets_init', 'swt_register_sidebars', 5 );

/* Add custom scripts. */
add_action( 'wp_enqueue_scripts', 'swt_enqueue_scripts' );

/* Register custom styles. */
add_action( 'wp_enqueue_scripts',    'swt_register_styles', 0 );

/* Filters the excerpt length. */
add_filter( 'excerpt_length', 'swt_excerpt_length' );

/* Remove read more in excerpt */
add_filter( 'excerpt_more', 'swt_read_more' );

/* Main theme layout */
add_filter( 'get_theme_layout', 'main_theme_layout' );

/* Disable sidebar if 1 column layout */
add_filter( 'sidebars_widgets', 'swt_disable_sidebars' );
add_action( 'template_redirect', 'swt_one_column' );	

/* Remove allowed tags from comment form */
add_filter( 'comment_form_defaults', 'remove_comment_form_allowed_tags' );

/* Add header scripts, custom css and favicon to <head> */
add_action( 'wp_head', 'swt_head_hooks' );

/* Add footer scripts to </body> */
add_action( 'wp_footer', 'swt_footer_scripts' ); 

/* Allow more tags in Options Framework */
add_action( 'admin_init', 'optionscheck_change_santiziation', 100 );

 
/**
 * Registers custom image sizes for the theme.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_image_sizes() {

	/* Sets the 'post-thumbnail' size. */
	set_post_thumbnail_size( 150, 150, true );

	/* SWT fp size */
	add_image_size( 'swt-fp', 370, 245, true );        	
		
}

/**
 * Registers nav menu locations.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_menus() {
	register_nav_menu( 'primary',    _x( 'Primary',    'nav menu location', 'swt' ) );
}

/**
 * Registers sidebars.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_sidebars() {
 
	hybrid_register_sidebar(
		array(
			'id'          => 'primary',
			'name'        => _x( 'Primary', 'sidebar', 'swt' ),
			'description' => __( 'The main sidebar. It is displayed on either the left or right side of the page based on the chosen layout.', 'swt' )
		)
	);

}
 

/**
 * Enqueues scripts.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_enqueue_scripts() {

	if ( !is_singular() && !is_404() ) {
		wp_enqueue_script( 'swt-imagesloaded', trailingslashit( get_template_directory_uri() ) . 'js/imagesloaded.pkgd.min.js', array( 'jquery' ), null, true );
		wp_enqueue_script( 'swt-masonry', trailingslashit( get_template_directory_uri() ) . 'js/masonry.pkgd.min.js', array( 'jquery' ), null, true );
		wp_enqueue_script( 'swt-infinite_scroll', trailingslashit( get_template_directory_uri() ). 'js/infinitescroll.min.js', array( 'jquery' ), null, true );
		wp_enqueue_script( 'wp-mediaelement' );			
		add_action( 'wp', 'custom_paged_404_fix' );				
		
	}

	wp_enqueue_script( 'swt-fitvids', trailingslashit( get_template_directory_uri() ) . 'js/fitvids.js', array( 'jquery' ), null, true );
	wp_enqueue_script( 'swt-responsive-nav', trailingslashit( get_template_directory_uri() ) . 'js/responsive-nav.min.js' );			
	wp_enqueue_script( 'swt-custom-js', trailingslashit( get_template_directory_uri() ) . 'js/custom.js', array( 'jquery' ), null, true );

	global $swt_fname, $swt_email;

	$swt_fname = __( 'Your Name', 'swt' );
	$swt_email = __( 'Email Address', 'swt' );	
	
	$params = array( 
		'finishedMsg' 	=> __( 'All posts loaded!', 'swt' ),
		'is_singular' 	=> is_singular(),
		'adding_msg'	=> __( 'Adding email address...', 'swt' ),
		'fname' 		=> $swt_fname,
		'email' 		=> $swt_email
	);
			
	wp_localize_script( 'swt-custom-js', 'options', $params ); 
	
}

function custom_paged_404_fix( ) {
	global $wp_query;

	if ( is_404() || !is_paged() || 0 != count( $wp_query->posts ) )
		return;

	$wp_query->set_404();
	status_header( 404 );
	nocache_headers();
}


/**
 * Registers custom stylesheets for the front end.
 *
 * @since  1.0.0
 * @access public
 * @return void
 */
function swt_register_styles() {
	
	/* Font Icon Font */
	wp_register_style( 'font-awesome', '//netdna.bootstrapcdn.com/font-awesome/4.2.0/css/font-awesome.css' );	
	wp_enqueue_style( 'font-awesome' );
	
	/* Theme Font */
	wp_register_style( 'font-merriweather', 'http://fonts.googleapis.com/css?family=Merriweather:700&subset=latin,latin-ext' );	
	wp_enqueue_style( 'font-merriweather' );	
	
    wp_enqueue_style( 'mediaelement' );
    wp_enqueue_style( 'wp-mediaelement' );	
}
 

/**
 * Adds a custom excerpt length.
 *
 * @since  1.0.0
 * @access public
 * @param  int     $length
 * @return int
 */
function swt_excerpt_length( $length ) {
	return 60;
}

function swt_read_more( $more ) {
	global $post;
	return '';	
}

/*
* Default layout
*/
function main_theme_layout( $layout ) {
			
	if ( !is_singular() ) {
		$layout = '';
	}
	elseif ( is_single() ) {
		if ( get_post_layout( get_the_ID() ) == 'default' ){
		        $layout = of_get_option( 'site_layout', 'layout-2c-l' );
		}
	}
	elseif ( is_page() ) {
		if ( get_post_layout( get_the_ID() ) == 'default' ){
		        $layout = of_get_option( 'site_layout', 'layout-2c-l' );
		}
	} 
	else {
		$layout = 'layout-'.get_post_layout( get_the_ID() );
	}    	

	return $layout;
}
  

/**
 * Disables sidebars if viewing a one-column page.
 *
 * @since 0.1.0
 */
function swt_disable_sidebars( $sidebars_widgets ) {
	
	global $wp_query;

	if ( current_theme_supports( 'theme-layouts' ) ) {

		if ( 'layout-1c' == theme_layouts_get_layout() ) {
			$sidebars_widgets['primary'] = false;
		}
                
	}

	return $sidebars_widgets;
}

/**
 * Function for deciding which pages should have a one-column layout.
 *
 * @since 0.1.0
 */ 
function swt_one_column() {

	if ( !is_active_sidebar( 'primary' ) || is_attachment() && wp_attachment_is_image() )
		add_filter( 'get_theme_layout', 'swt_theme_layout_one_column' );

	elseif ( is_attachment() && 'layout-default' == theme_layouts_get_layout() )
		add_filter( 'get_theme_layout', 'swt_theme_layout_one_column' );
}

/**
 * Filters 'get_theme_layout' by returning 'layout-1c'.
 *
 * @since 0.2.0
 */
function swt_theme_layout_one_column( $layout ) {
	return 'layout-1c';
}

/* Disables comments allowed tags */
function remove_comment_form_allowed_tags( $defaults ) {
 
	$defaults['comment_notes_before'] = '';
	$defaults['comment_notes_after'] = '';
	return $defaults;
	 
}

/* Functions from theme options - header scripts, custom css and favicon */
function swt_head_hooks() {
	
	$header_scripts = of_get_option( 'header_scripts' ); 
	$custom_css = of_get_option( 'custom_css' );
	$favicon = of_get_option( 'favicon' ); 
	
	if ( $favicon ) { 
		echo "<link rel=\"icon\" href=\"{$favicon}\" type=\"image/x-icon\" />\n";
	}		
	
	if ( $header_scripts ) 
		echo $header_scripts ."\n";
	
	if ( $custom_css ) 
		echo "<style type=\"text/css\">\n". $custom_css ."\n</style>\n";
}

/* Adds footer scripts */
function swt_footer_scripts() {
	
	$footer_scripts = of_get_option( 'footer_scripts' ); 
	
	if ( $footer_scripts )
		echo $footer_scripts . "\n";

}

/*this function allows for the auto-creation of post excerpts*/
function truncate_post( $amount,$quote_after=false ) {
	$truncate = get_the_content();
	$truncate = apply_filters('the_content', $truncate);
	$truncate = preg_replace('@<script[^>]*?>.*?</script>@si', '', $truncate);
	$truncate = preg_replace('@<style[^>]*?>.*?</style>@si', '', $truncate);
	$truncate = strip_tags($truncate);
	$truncate = substr($truncate, 0, strrpos(substr($truncate, 0, $amount), ' '));
	echo $truncate;
	echo "...";
	if ($quote_after) echo('');
} 

/* Change sanitize options for Options Framework */
function optionscheck_change_santiziation() {
    remove_filter( 'of_sanitize_textarea', 'of_sanitize_textarea' );
    add_filter( 'of_sanitize_textarea', 'custom_sanitize_textarea' );
}

function custom_sanitize_textarea( $input ) {

    global $allowedposttags;
	
	$custom_allowedtags["script"] = 
		array(
			"text" => array(),
			"type" => array()
		);
	
	$custom_allowedtags = array_merge( $custom_allowedtags, $allowedposttags );
	$output = wp_kses( $input, $custom_allowedtags );
	
	return $output;
}